using System;
using System.Diagnostics;
using System.Threading.Tasks;

namespace Implab {
    public class Deferred<T> : IResolvable<T> {
        readonly Promise<T> m_promise;

        internal Deferred() {
            m_promise = new Promise<T>();
        }

        protected Deferred(Promise<T> promise) {
            Debug.Assert(promise != null);
            m_promise = promise;
        }

        public IPromise<T> Promise {
            get { return m_promise; }
        }

        public void Cancel() {
            Reject(new OperationCanceledException());
        }

        public virtual void Reject(Exception error) {
            if (error is PromiseTransientException)
                error = ((PromiseTransientException)error).InnerException;

            m_promise.RejectPromise(error);
        }

        public virtual void Resolve(T value) {
            m_promise.ResolvePromise(value);
        }

        public virtual void Resolve(IPromise<T> thenable) {
            if (thenable == null)
                Reject(new Exception("The promise or task are expected"));
            if (thenable == m_promise)
                Reject(new Exception("The promise cannot be resolved with oneself"));

            try {
                thenable.Then(this);
            } catch (Exception err) {
                Reject(err);
            }
        }

        public virtual void Resolve(Task<T> thenable) {
            if (thenable == null)
                Reject(new Exception("The promise or task are expected"));
            
            try {
                thenable.Then(this);
            } catch (Exception err) {
                Reject(err);
            }
        }
    }
}